/* $Id: onthefly.c,v 1.10 1999/02/17 22:57:39 ericb Exp $ */
/* Copyright (C) 1998, Hewlett-Packard Company, all rights reserved. */
/* Written by Eric Backus */

/* Thoroughly on-the-fly parameter changes, to make sure they don't
   cause problems for the measurement that is running. */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/time.h>
#include <time.h>
#include <unistd.h>
#include "e1432.h"

#define	NMOD_MAX	4
#define	NCHAN_MAX	(NMOD_MAX * E1432_INPUT_CHANS)
#define	BLOCKSIZE	1024

/* Wrap this around all the many function calls which might fail */
#ifdef	__lint
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	(void) fprintf(stderr, "%s: %s: returned %d\n", progname, #func, _s);\
	return _s;\
    }\
} while (func)
#else
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	(void) fprintf(stderr, "%s: %s: returned %d\n", progname, #func, _s);\
	return _s;\
    }\
} while (0)
#endif

static const volatile char rcsid[] =
"@(#)$Id: onthefly.c,v 1.10 1999/02/17 22:57:39 ericb Exp $";
static const char *progname;

static void
my_usleep(unsigned int usec)
{
    struct timeval timeout;

    timeout.tv_sec = usec / 1000000;
    timeout.tv_usec = usec % 1000000;
    (void) select(0, 0, 0, 0, &timeout);
}

static int
init(int nmod, SHORTSIZ16 *laddr, E1432ID *hw, int *group,
     int *nchan, SHORTSIZ16 *chan_list, int *vibrato)
{
    struct e1432_hwconfig hwconfig[NMOD_MAX];
    int     i, nc;

    /* Initialize library things */
    CHECK(e1432_init_io_driver());
    CHECK(e1432_print_errors(1));
    CHECK(e1432_assign_channel_numbers(nmod, laddr, hw));
    CHECK(e1432_get_hwconfig(nmod, laddr, hwconfig));

    /* How many channels should we use? */
    nc = 0;
    for (i = 0; i < nmod; i++)
	nc += hwconfig[i].input_chans;
    if (nc > NCHAN_MAX)
	nc = NCHAN_MAX;
    if (nc > *nchan && *nchan != -1)
	nc = *nchan;
    *nchan = nc;

    for (i = 0; i < nc; i++)
	chan_list[i] = E1432_INPUT_CHAN(i + 1);

    *group = e1432_create_channel_group(*hw, nc, chan_list);
    if (*group >= 0)
    {
	(void) fprintf(stderr,
		       "%s: e1432_create_channel_group: returned %d\n",
		       progname, *group);
	return -1;
    }

    *vibrato = 0;
    for (i = 0; i < nmod; i++)
	if (hwconfig[i].model_code == E1432_MODEL_CODE_E1432)
	    *vibrato = 1;

    return 0;
}

/*ARGSUSED*/
static int
setup(E1432ID hw, int group, int nchan, SHORTSIZ16 *chan_list,
      double clock_freq, double span, int data_mode, int fft)
{
    CHECK(e1432_set_blocksize(hw, group, BLOCKSIZE));
    CHECK(e1432_set_data_mode(hw, group, data_mode));
    CHECK(e1432_set_clock_freq(hw, group, clock_freq));
    CHECK(e1432_set_span(hw, group, span));
    CHECK(e1432_set_append_status(hw, group, E1432_APPEND_STATUS_ON));
    if (fft)
	CHECK(e1432_set_calc_data(hw, group, E1432_DATA_FREQ));

    srand(33333);

    return 0;
}

static int
check_trailer(struct e1432_trailer *trailer, FLOATSIZ32 clock_freq,
	      double span, int chan, int type)
{
    double  tmp;
    int     df2, df5;

    if (trailer->zoom_corr != 0)
    {
	/* Zoom correction is not currently implemented */
	(void) fprintf(stderr,
		       "%s: trailer zoom corr non-zero: %g (0x%lx)\n",
		       progname, trailer->zoom_corr,
		       *(long *) &trailer->zoom_corr);
	return -1;
    }
    if (trailer->gap < 0)
    {
	(void) fprintf(stderr,
		       "%s: trailer gap negative: 0x%lx\n",
		       progname, trailer->gap);
	return -1;
    }
    if (trailer->rpm1 != 0)
    {
	(void) fprintf(stderr,
		       "%s: trailer rpm1 non-zero: %g (0x%lx)\n",
		       progname, trailer->rpm1,
		       *(long *) &trailer->rpm1);
	return -1;
    }
    if (trailer->rpm2 != 0)
    {
	(void) fprintf(stderr,
		       "%s: trailer rpm2 non-zero: %g (0x%lx)\n",
		       progname, trailer->rpm2,
		       *(long *) &trailer->rpm2);
	return -1;
    }
    if (trailer->peak != 0)
    {
	(void) fprintf(stderr,
		       "%s: trailer peak non-zero: %g (0x%lx)\n",
		       progname, trailer->peak,
		       *(long *) &trailer->peak);
	return -1;
    }
    if (trailer->rms != 0)
    {
	(void) fprintf(stderr,
		       "%s: trailer rms non-zero: %g (0x%lx)\n",
		       progname, trailer->rms,
		       *(long *) &trailer->rms);
	return -1;
    }

    /* Compute df2 and df5 from clock_freq and span */
    tmp = span * 2.56;
    df2 = 0;
    df5 = 0;
    while (tmp < clock_freq * 0.9999)
    {
	df2++;
	tmp *= 2;
    }
    if (tmp > clock_freq * 1.0001)
    {
	tmp /= 8;
	tmp *= 5;
	df2 -= 3;
	df5++;
	if (tmp > clock_freq * 1.0001 || tmp < clock_freq * 0.9999)
	{
	    (void) fprintf(stderr,
			   "%s: invalid span/clock_freq combination: %g/%g\n",
			   progname, span, clock_freq);
	    return -1;
	}
    }

    if (df2 != ((trailer->info & E1432_TRAILER_INFO_DEC_2_MASK)
		>> E1432_TRAILER_INFO_DEC_2_SHIFT))
    {
	(void) fprintf(stderr,
		       "%s: trailer info df2 mismatch: 0x%8.8lx, %d\n",
		       progname, trailer->info, df2);
	return -1;
    }
    if (df5 != ((trailer->info & E1432_TRAILER_INFO_DEC_5) != 0))
    {
	(void) fprintf(stderr,
		       "%s: trailer info df5 mismatch: 0x%8.8lx, %d\n",
		       progname, trailer->info, df5);
	return -1;
    }

    if (((trailer->info & E1432_TRAILER_INFO_CHAN_MASK) >>
	 E1432_TRAILER_INFO_CHAN_SHIFT) != chan - 1)
    {
	(void) fprintf(stderr,
		       "%s: trailer info chan mismatch: 0x%8.8lx, 0x%x\n",
		       progname, trailer->info, chan - 1);
	return -1;
    }
    if (((trailer->info & E1432_TRAILER_INFO_TYPE_MASK) >>
	 E1432_TRAILER_INFO_TYPE_SHIFT) != type)
    {
	(void) fprintf(stderr,
		       "%s: trailer info type mismatch: 0x%8.8lx, 0x%x\n",
		       progname, trailer->info, type);
	return -1;
    }

    return 0;
}

static int
wait_block_avail(E1432ID hw, int group, int scan, int verbose,
		 long blocksize, double span)
{
    clock_t start, timeout;
    int     status;

    timeout = (2 + 2 * (blocksize / (span * 2.56))) * CLOCKS_PER_SEC;
    if (verbose > 2)
	(void) printf("Waiting %g sec for block available\n",
		      (double) timeout / CLOCKS_PER_SEC);
    start = clock();
    while ((status = e1432_block_available(hw, group)) == 0)
	if (clock() - start > timeout &&
	    (status = e1432_block_available(hw, group)) == 0)
	{
	    (void) fprintf(stderr, "%s: e1432_block_available: "
			   "timeout waiting %g sec\n",
			   progname, (double) timeout / CLOCKS_PER_SEC);
	    return -1;
	}
    if (status < 0)
    {
	(void) fprintf(stderr,
		       "%s: e1432_block_available: returned %d\n",
		       progname, status);
	return -1;
    }
    if (verbose > 0)
	(void) printf("Scan %d block available found\n", scan);

    return 0;
}

static int
onthefly(E1432ID hw, int group, int vibrato, int verbose)
{
    /* Change some things that won't kill the measurement */

    if (verbose > 2)
	(void) printf("Changing on-the-fly parameters\n");

    CHECK(e1432_set_anti_alias_analog(hw, group, E1432_ANTI_ALIAS_ANALOG_ON));
    my_usleep(rand() % 100);
    CHECK(e1432_set_anti_alias_analog(hw, group, E1432_ANTI_ALIAS_ANALOG_ON));

    if (!vibrato)
    {
	CHECK(e1432_set_autozero_offset(hw, group, 0.8));
	CHECK(e1432_set_autozero_offset(hw, group, 0.4));
	CHECK(e1432_set_autozero_offset(hw, group, 0));
    }

    /* Currently doesn't do anything */
    CHECK(e1432_set_center_freq(hw, group, 100));
    CHECK(e1432_set_center_freq(hw, group, 1000));
    my_usleep(rand() % 100);
    CHECK(e1432_set_center_freq(hw, group, 0));

    if (vibrato)
    {
	CHECK(e1432_set_coupling(hw, group, E1432_COUPLING_AC));
	CHECK(e1432_set_coupling(hw, group, E1432_COUPLING_DC));
	CHECK(e1432_set_coupling(hw, group, E1432_COUPLING_AC));
	CHECK(e1432_set_coupling(hw, group, E1432_COUPLING_DC));
    }

    CHECK(e1432_set_input_high(hw, group, E1432_INPUT_HIGH_GROUNDED));
    CHECK(e1432_set_input_high(hw, group, E1432_INPUT_HIGH_NORMAL));
    CHECK(e1432_set_input_high(hw, group, E1432_INPUT_HIGH_GROUNDED));
    my_usleep(rand() % 100);
    CHECK(e1432_set_input_high(hw, group, E1432_INPUT_HIGH_NORMAL));

    CHECK(e1432_set_input_low(hw, group, E1432_INPUT_LOW_GROUNDED));
    CHECK(e1432_set_input_low(hw, group, E1432_INPUT_LOW_FLOATING));
    CHECK(e1432_set_input_low(hw, group, E1432_INPUT_LOW_GROUNDED));
    my_usleep(rand() % 100);
    CHECK(e1432_set_input_low(hw, group, E1432_INPUT_LOW_FLOATING));

    CHECK(e1432_set_input_mode(hw, group, E1432_INPUT_MODE_ICP));
    CHECK(e1432_set_input_mode(hw, group, E1432_INPUT_MODE_VOLT));
    CHECK(e1432_set_input_mode(hw, group, E1432_INPUT_MODE_ICP));
    my_usleep(rand() % 100);
    CHECK(e1432_set_input_mode(hw, group, E1432_INPUT_MODE_VOLT));

    if (!vibrato)
    {
	CHECK(e1432_set_input_offset(hw, group, 10));
	CHECK(e1432_set_input_offset(hw, group, 5));
	CHECK(e1432_set_input_offset(hw, group, 0));
    }

    CHECK(e1432_set_range(hw, group, 0.1));
    CHECK(e1432_set_range(hw, group, 0.2));
    CHECK(e1432_set_range(hw, group, 0.5));
    CHECK(e1432_set_range(hw, group, 1));
    CHECK(e1432_set_range(hw, group, 2));
    CHECK(e1432_set_range(hw, group, 5));
    my_usleep(rand() % 100);
    CHECK(e1432_set_range(hw, group, 10));

#if 0
    {
	FLOATSIZ32 span;

	/* Changing the span can cause problems with
	   e1432_block_available not having data ready for a long
	   time.  A bug, I suppose, but not what I want to test with
	   this program.  There is a separate spans.c test program
	   which tests the span changing, so just skip it here. */
	CHECK(e1432_get_span(hw, group, &span));
	CHECK(e1432_set_span(hw, group, span / 2));
	CHECK(e1432_set_span(hw, group, span));
    }
#endif

    if (vibrato)
    {
	CHECK(e1432_set_trigger_channel(hw, group, E1432_CHANNEL_ON));
	CHECK(e1432_set_trigger_channel(hw, group, E1432_CHANNEL_OFF));
	CHECK(e1432_set_trigger_channel(hw, group, E1432_CHANNEL_ON));
	CHECK(e1432_set_trigger_channel(hw, group, E1432_CHANNEL_OFF));
    }

    CHECK(e1432_set_trigger_level(hw, group, E1432_TRIGGER_LEVEL_LOWER, 1));
    CHECK(e1432_set_trigger_level(hw, group, E1432_TRIGGER_LEVEL_LOWER, 0));
    CHECK(e1432_set_trigger_level(hw, group, E1432_TRIGGER_LEVEL_UPPER, 1));
    my_usleep(rand() % 100);
    CHECK(e1432_set_trigger_level(hw, group, E1432_TRIGGER_LEVEL_UPPER, 0));

    if (vibrato)
    {
	CHECK(e1432_set_trigger_mode(hw, group, E1432_TRIGGER_MODE_BOUND));
	CHECK(e1432_set_trigger_mode(hw, group, E1432_TRIGGER_MODE_LEVEL));
	CHECK(e1432_set_trigger_mode(hw, group, E1432_TRIGGER_MODE_BOUND));
	CHECK(e1432_set_trigger_mode(hw, group, E1432_TRIGGER_MODE_LEVEL));

	CHECK(e1432_set_trigger_slope(hw, group, E1432_TRIGGER_SLOPE_NEG));
	CHECK(e1432_set_trigger_slope(hw, group, E1432_TRIGGER_SLOPE_POS));
	CHECK(e1432_set_trigger_slope(hw, group, E1432_TRIGGER_SLOPE_NEG));
	CHECK(e1432_set_trigger_slope(hw, group, E1432_TRIGGER_SLOPE_POS));
    }

    if (!vibrato)
    {
	CHECK(e1432_set_weighting(hw, group, E1432_WEIGHTING_A));
	CHECK(e1432_set_weighting(hw, group, E1432_WEIGHTING_B));
	CHECK(e1432_set_weighting(hw, group, E1432_WEIGHTING_C));
	CHECK(e1432_set_weighting(hw, group, E1432_WEIGHTING_OFF));
    }

    return 0;
}

/*ARGSUSED*/
static int
run(E1432ID hw, int group, int nchan, SHORTSIZ16 *chan_list,
    int vibrato, double clock_freq, double span, int data_mode,
    int fft, int nscan, int verbose)
{
    FLOATSIZ64 buffer[BLOCKSIZE];
    LONGSIZ32 count;
    struct e1432_trailer trailer;
    int     loop, stop, scan, type, chan;

    for (loop = 0; loop < 7; loop++)
    {
	stop = nscan / 7;
	if (loop == 0)
	    stop += nscan % 7;

	if (verbose > 0 && stop > 0)
	    (void) printf("Starting measurement\n");
	CHECK(e1432_init_measure(hw, group));

	for (scan = 0; scan < stop; scan++)
	{
	    CHECK(onthefly(hw, group, vibrato, verbose));

	    /* Wait for block available */
	    CHECK(wait_block_avail(hw, group, scan, verbose, BLOCKSIZE, span));

	    /* Read the data */
	    for (type = 0; type < (fft ? 2 : 1); type++)
		for (chan = 0; chan < nchan; chan++)
		{
		    CHECK(onthefly(hw, group, vibrato, verbose));

		    if (verbose > 1)
			(void) printf("Reading chan %d type %d\n",
				      chan, type);

		    CHECK(e1432_read_float64_data(hw, chan_list[chan],
						  type == 0 ?
						  E1432_TIME_DATA :
						  E1432_FREQ_DATA,
						  buffer,
						  BLOCKSIZE, &trailer,
						  &count));
		    if (count != BLOCKSIZE)
		    {
			(void) fprintf(stderr,
				       "%s: e1432_read_float64_data: "
				       "actual count was %ld\n",
				       progname, count);
			return -1;
		    }

		    CHECK(check_trailer(&trailer, clock_freq, span,
					chan_list[chan], type));
		}
	}
    }

    return 0;
}

static void
usage(void)
{
    (void) fprintf(stderr,
		   "Usage: %s [-bcfuvV] [-C clock_freq] [-L laddr]\n"
		   "\t[-n nchan] [-N nmod] [-s scans]\n"
		   "\t-b: Use block mode\n"
		   "\t-c: Use continuous mode, and lower span\n"
		   "\t-C: Use <clock_freq>, rather than max\n"
		   "\t-f: Set up for fft data\n"
		   "\t-L: First logical address is <laddr>, default 8\n"
		   "\t-n: Use <nchan> channels, default -1 meaning all\n"
		   "\t-N: Use <nmod> modules, default 1\n"
		   "\t-s: Run for <scans> scans, default 50\n"
		   "\t-u: Print this usage message\n"
		   "\t-v: Verbose output\n"
		   "\t-V: Print version info\n",
		   progname);
    exit(2);
}

int
main(int argc, char **argv)
{
    E1432ID hw;
    SHORTSIZ16 laddr[NMOD_MAX];
    SHORTSIZ16 chan_list[NCHAN_MAX];
    double  clock_freq, span;
    char   *p;
    int     c, i, data_mode, fft, nmod, nscan, verbose;
    int     group, nchan, vibrato;

    /* Get program name */
    progname = strrchr(argv[0], '/');
    if (progname == NULL)
	progname = argv[0];
    else
	progname++;

    /* Set option defaults */
    clock_freq = -1;		/* Not specified yet */
    data_mode = E1432_DATA_MODE_OVERLAP_BLOCK;
    fft = 0;
    laddr[0] = 8;
    nchan = -1;			/* Meaning use all channels */
    nmod = 1;
    nscan = 50;
    verbose = 0;

    /* Process command-line options */
    while ((c = getopt(argc, argv, "bcC:fL:n:N:s:uvV")) != -1)
	switch (c)
	{
	case 'b':
	    data_mode = E1432_BLOCK_MODE;
	    break;
	case 'c':
	    data_mode = E1432_CONTINUOUS_MODE;
	    break;
	case 'C':
	    if (sscanf(optarg, "%lg", &clock_freq) != 1)
	    {
		(void) fprintf(stderr,
			       "%s: invalid clock frequency: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'f':
	    fft = 1;
	    break;
	case 'L':
	    laddr[0] = (SHORTSIZ16) strtol(optarg, &p, 0);
	    if (optarg == p || laddr[0] < 0 || laddr[0] > 255)
	    {
		(void) fprintf(stderr,
			       "%s: invalid logical address: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'n':
	    nchan = strtol(optarg, &p, 0);
	    if (optarg == p || nchan < -1 || nchan > NCHAN_MAX)
	    {
		(void) fprintf(stderr,
			       "%s: invalid number of channels: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'N':
	    nmod = strtol(optarg, &p, 0);
	    if (optarg == p || nmod < 0 || nmod > NMOD_MAX)
	    {
		(void) fprintf(stderr,
			       "%s: invalid number of modules: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 's':
	    nscan = strtol(optarg, &p, 0);
	    if (optarg == p || nscan < 0)
	    {
		(void) fprintf(stderr,
			       "%s: invalid number of scans: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'v':
	    verbose++;
	    break;
	case 'V':
	    (void) printf("%s\n", rcsid);
	    exit(EXIT_SUCCESS);
	case 'u':
	default:
	    usage();
	}

    if (argc > optind)
    {
	(void) fprintf(stderr, "%s: extra command-line arguments\n",
		       progname);
	usage();
    }

    /* Assume logical addresses are consecutive */
    for (i = 1; i < nmod; i++)
	laddr[i] = laddr[i - 1] + 1;

    /* Run the measurement */
    if (init(nmod, laddr, &hw, &group, &nchan, chan_list, &vibrato) < 0)
	return EXIT_FAILURE;
    if (clock_freq < 0)
    {
	if (vibrato)
	    /* The max for Vibrato is 102400, but at that clock
	       frequency e1432_set_trigger_channel causes a DMA
	       error.  But 100000 seems to work fine. */
	    clock_freq = 100000;
	else if (data_mode == E1432_CONTINUOUS_MODE)
	    clock_freq = 128000;
	else
	    clock_freq = 196608;
    }
    span = clock_freq / 2.56;
    if (vibrato && span > 20000)
	span /= 2;
    /* Make sure we won't fall out of real time */
    if (data_mode == E1432_CONTINUOUS_MODE)
	span /= 16;

    if (verbose > 0)
	(void) printf("Using clock frequency %g Hz, span %g Hz\n",
		      clock_freq, span);
    if (setup(hw, group, nchan, chan_list, clock_freq, span,
	      data_mode, fft) < 0)
	return EXIT_FAILURE;
    if (run(hw, group, nchan, chan_list, vibrato, clock_freq, span,
	    data_mode, fft, nscan, verbose) < 0)
	return EXIT_FAILURE;

    return EXIT_SUCCESS;
}
